/*
 * Copyright 2015 Licel Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package pro.javacard.engine.proxy.javacard.security;

import com.licel.jcardsim.crypto.AsymmetricSignatureImpl;
import com.licel.jcardsim.crypto.SymmetricSignatureImpl;
import javacard.security.CryptoException;
import javacard.security.Key;
import javacard.security.MessageDigest;
import javacard.security.Signature;
import javacardx.crypto.Cipher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * ProxyClass for <code>Signature</code>
 *
 * @see Signature
 */
public class SignatureProxy {
    private static final Logger log = LoggerFactory.getLogger(SignatureProxy.class);

    /**
     * Creates a <code>Signature</code> object instance of the selected algorithm.
     *
     * @param algorithm      the desired Signature algorithm. Valid codes listed in
     *                       ALG_ .. constants above e.g. <A HREF="../../javacard/security/Signature.html#ALG_DES_MAC4_NOPAD"><CODE>ALG_DES_MAC4_NOPAD</CODE></A>
     * @param externalAccess <code>true</code> indicates that the instance will be shared among
     *                       multiple applet instances and that the <code>Signature</code> instance will also be accessed (via a <code>Shareable</code>
     *                       interface) when the owner of the <code>Signature</code> instance is not the currently selected applet.
     *                       If <code>true</code> the implementation must not allocate CLEAR_ON_DESELECT transient space for internal data.
     * @return the <code>Signature</code> object instance of the requested algorithm
     * @throws CryptoException with the following reason codes:<ul>
     *                         <li><code>CryptoException.NO_SUCH_ALGORITHM</code> if the requested algorithm
     *                         or shared access mode is not supported.</ul>
     */
    public static final Signature getInstance(byte algorithm, boolean externalAccess)
            throws CryptoException {
        Signature instance = null;
        //TODO: implement externalAccess logic
//        if (externalAccess) {
//            CryptoException.throwIt(CryptoException.NO_SUCH_ALGORITHM);
//        }
        switch (algorithm) {
            case Signature.ALG_RSA_SHA_ISO9796:
            case Signature.ALG_RSA_SHA_PKCS1:
            case Signature.ALG_RSA_SHA_224_PKCS1:
            case Signature.ALG_RSA_SHA_256_PKCS1:
            case Signature.ALG_RSA_SHA_384_PKCS1:
            case Signature.ALG_RSA_SHA_512_PKCS1:
            case Signature.ALG_RSA_SHA_PKCS1_PSS:
            case Signature.ALG_RSA_SHA_224_PKCS1_PSS:
            case Signature.ALG_RSA_SHA_256_PKCS1_PSS:
            case Signature.ALG_RSA_SHA_384_PKCS1_PSS:
            case Signature.ALG_RSA_SHA_512_PKCS1_PSS:
            case Signature.ALG_RSA_MD5_PKCS1:
            case Signature.ALG_RSA_RIPEMD160_ISO9796:
            case Signature.ALG_RSA_RIPEMD160_PKCS1:
            case Signature.ALG_ECDSA_SHA:
            case Signature.ALG_ECDSA_SHA_224:
            case Signature.ALG_ECDSA_SHA_256:
            case Signature.ALG_ECDSA_SHA_384:
            case Signature.ALG_ECDSA_SHA_512:
            case Signature.ALG_RSA_SHA_ISO9796_MR:
            case Signature.ALG_DSA_SHA:
            case Signature.ALG_RSA_SHA_RFC2409:
            case Signature.ALG_RSA_MD5_RFC2409:
            case Signature.ALG_RSA_MD5_PKCS1_PSS:
            case Signature.ALG_RSA_RIPEMD160_PKCS1_PSS:
            case Signature.ALG_RSA_RIPEMD160_ISO9796_MR:
                try {
                    instance = new AsymmetricSignatureImpl(algorithm);
                } catch (Exception e) {
                    log.error("getInstance of asymmetric algo: " + algorithm + " is NOT OK! : {}", e.getClass().getSimpleName());
                    CryptoException.throwIt(CryptoException.INVALID_INIT);
                }
                break;
            case Signature.ALG_DES_MAC4_NOPAD:
            case Signature.ALG_DES_MAC8_NOPAD:
            case Signature.ALG_DES_MAC4_ISO9797_M1:
            case Signature.ALG_DES_MAC8_ISO9797_M1:
            case Signature.ALG_DES_MAC4_ISO9797_M2:
            case Signature.ALG_DES_MAC8_ISO9797_M2:
            case Signature.ALG_DES_MAC8_ISO9797_1_M2_ALG3:
            case Signature.ALG_DES_MAC4_PKCS5:
            case Signature.ALG_DES_MAC8_PKCS5:
            case Signature.ALG_AES_MAC_128_NOPAD:
            case Signature.ALG_HMAC_SHA1:
            case Signature.ALG_HMAC_SHA_256:
            case Signature.ALG_HMAC_SHA_384:
            case Signature.ALG_HMAC_SHA_512:
            case Signature.ALG_HMAC_MD5:
            case Signature.ALG_HMAC_RIPEMD160:
            case Signature.ALG_AES_CMAC_128:
                instance = new SymmetricSignatureImpl(algorithm);
                break;

            default:
                CryptoException.throwIt(CryptoException.NO_SUCH_ALGORITHM);
                break;
        }
        return instance;
    }

    public static final Signature getInstance(byte messageDigestAlgorithm, byte cipherAlgorithm,
                                              byte paddingAlgorithm, boolean externalAccess) throws CryptoException {

        // TODO: codify the mapping against BC
        switch (cipherAlgorithm) {
            case Signature.SIG_CIPHER_AES_CMAC128:
                if (messageDigestAlgorithm != MessageDigest.ALG_NULL)
                    CryptoException.throwIt(CryptoException.NO_SUCH_ALGORITHM);
                if (paddingAlgorithm != Cipher.PAD_ISO9797_M2)
                    CryptoException.throwIt(CryptoException.NO_SUCH_ALGORITHM);
                return new SymmetricSignatureImpl(Signature.ALG_AES_CMAC_128); // FIXME: need padding
            case Signature.SIG_CIPHER_ECDSA:
                switch (messageDigestAlgorithm) {
                    case MessageDigest.ALG_SHA_256:
                        return new AsymmetricSignatureImpl(Signature.ALG_ECDSA_SHA_256);
                    case MessageDigest.ALG_SHA_384:
                        return new AsymmetricSignatureImpl(Signature.ALG_ECDSA_SHA_384);
                    case MessageDigest.ALG_SHA_512:
                        return new AsymmetricSignatureImpl(Signature.ALG_ECDSA_SHA_512);
                    default:
                        CryptoException.throwIt(CryptoException.NO_SUCH_ALGORITHM);
                }
            default:
                CryptoException.throwIt(CryptoException.NO_SUCH_ALGORITHM);
        }
        // Not reached
        log.error("Not reachable.");
        return null;
    }

    public static final class OneShot extends Signature {
        private static final Logger log = LoggerFactory.getLogger(OneShot.class);
        private Signature signature;

        private OneShot() {
            log.debug("Signature.OneShot");
        }

        public static SignatureProxy.OneShot open(byte messageDigestAlgorithm, byte cipherAlgorithm, byte paddingAlgorithm) {
            SignatureProxy.OneShot one = new SignatureProxy.OneShot();
            one.signature = Signature.getInstance(messageDigestAlgorithm, cipherAlgorithm, paddingAlgorithm, false);
            return one;
        }

        @Override
        public void init(Key key, byte b) throws CryptoException {
            signature.init(key, b);
        }

        @Override
        public void init(Key key, byte b, byte[] bytes, short i, short i1) throws CryptoException {
            signature.init(key, b, bytes, i, i1);
        }

        @Override
        public void setInitialDigest(byte[] bytes, short i, short i1, byte[] bytes1, short i2, short i3) throws CryptoException {
            signature.setInitialDigest(bytes, i, i1, bytes1, i2, i3);
        }

        @Override
        public byte getAlgorithm() {
            return signature.getAlgorithm();
        }

        @Override
        public byte getMessageDigestAlgorithm() {
            return signature.getMessageDigestAlgorithm();
        }

        @Override
        public byte getCipherAlgorithm() {
            return signature.getCipherAlgorithm();
        }

        @Override
        public byte getPaddingAlgorithm() {
            return signature.getPaddingAlgorithm();
        }

        @Override
        public short getLength() throws CryptoException {
            return signature.getLength();
        }

        @Override
        public void update(byte[] bytes, short i, short i1) throws CryptoException {
            CryptoException.throwIt(CryptoException.ILLEGAL_USE);
        }

        @Override
        public short sign(byte[] bytes, short i, short i1, byte[] bytes1, short i2) throws CryptoException {
            return signature.sign(bytes, i, i1, bytes1, i2);
        }

        @Override
        public short signPreComputedHash(byte[] bytes, short i, short i1, byte[] bytes1, short i2) throws CryptoException {
            return signature.signPreComputedHash(bytes, i, i1, bytes1, i2);
        }

        @Override
        public boolean verify(byte[] bytes, short i, short i1, byte[] bytes1, short i2, short i3) throws CryptoException {
            return signature.verify(bytes, i, i1, bytes1, i2, i3);
        }

        @Override
        public boolean verifyPreComputedHash(byte[] bytes, short i, short i1, byte[] bytes1, short i2, short i3) throws CryptoException {
            return signature.verifyPreComputedHash(bytes, i, i1, bytes1, i2, i3);
        }

        public void close() {
            signature = null;
        }
    }
}
