/*
 * Copyright 2011 Licel LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.licel.jcardsim.base;

import javacard.framework.JCSystem;
import javacard.framework.SystemException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Basic implementation of storage transient memory of JCRE.
 */
public class TransientMemory {
    private static final Logger log = LoggerFactory.getLogger(TransientMemory.class);
    /**
     * List of <code>CLEAR_ON_DESELECT</code> arrays
     */
    protected final ArrayList<Object> clearOnDeselect = new ArrayList<>();
    /**
     * List of <code>CLEAR_ON_RESET</code> arrays
     */
    protected final ArrayList<Object> clearOnReset = new ArrayList<>();

    private int sumCOD;
    private int sumCOR;

    private void add(Object obj, byte event) {
        int toAdd = 0;
        if (obj instanceof byte[]) {
            toAdd += ((byte[]) obj).length;
        } else if (obj instanceof short[]) {
            toAdd += ((short[]) obj).length * 2;
        } else if (obj instanceof Object[]) {
            // Assume 16 bits for pointer. Arbitrary
            toAdd += ((Object[]) obj).length * 2;
        } else if (obj instanceof boolean[]) {
            toAdd += ((boolean[]) obj).length;
        } else {
            log.warn("Unsupported object: {}", obj.getClass());
        }

        if (event == JCSystem.CLEAR_ON_RESET) {
            sumCOR += toAdd;
        } else if (event == JCSystem.CLEAR_ON_DESELECT) {
            sumCOD += toAdd;
        } else {
            log.warn("Unsupported event {}", event);
        }
    }

    /**
     * @param length the length of the array
     * @param event  the <code>CLEAR_ON...</code> event which causes the array elements to be cleared
     * @return the new transient array
     * @see javacard.framework.JCSystem#makeTransientBooleanArray(short, byte)
     */
    public boolean[] makeBooleanArray(short length, byte event) {
        boolean[] array = new boolean[length];
        storeArray(array, event);
        return array;
    }

    /**
     * @param length the length of the array
     * @param event  the <code>CLEAR_ON...</code> event which causes the array elements to be cleared
     * @return the new transient array
     * @see javacard.framework.JCSystem#makeTransientByteArray(short, byte)
     */
    public byte[] makeByteArray(int length, byte event) {
        byte[] array = new byte[length];
        storeArray(array, event);
        return array;
    }

    /**
     * @param length the length of the array
     * @param event  the <code>CLEAR_ON...</code> event which causes the array elements to be cleared
     * @return the new transient array
     * @see javacard.framework.JCSystem#makeTransientShortArray(short, byte)
     */
    public short[] makeShortArray(short length, byte event) {
        short[] array = new short[length];
        storeArray(array, event);
        return array;
    }

    /**
     * @param length the length of the array
     * @param event  the <code>CLEAR_ON...</code> event which causes the array elements to be cleared
     * @return the new transient array
     * @see javacard.framework.JCSystem#makeTransientObjectArray(short, byte)
     */
    public Object[] makeObjectArray(short length, byte event) {
        Object[] array = new Object[length];
        storeArray(array, event);
        return array;
    }

    /**
     * @param type   the array type - must be one of : ARRAY_TYPE_BOOLEAN, ARRAY_TYPE_BYTE, ARRAY_TYPE_SHORT, ARRAY_TYPE_INT, or ARRAY_TYPE_OBJECT
     * @param length the length of the global transient array
     * @return the new transient Object array
     * @see javacard.framework.JCSystem#makeGlobalArray(byte, short)
     */
    public Object makeGlobalArray(byte type, short length) {
        Object array = null;
        switch (type) {
            case JCSystem.ARRAY_TYPE_BOOLEAN:
                array = makeBooleanArray(length, JCSystem.CLEAR_ON_RESET);
                break;
            case JCSystem.ARRAY_TYPE_BYTE:
                array = makeByteArray(length, JCSystem.CLEAR_ON_RESET);
                break;
            case JCSystem.ARRAY_TYPE_SHORT:
                array = makeShortArray(length, JCSystem.CLEAR_ON_RESET);
                break;
            case JCSystem.ARRAY_TYPE_OBJECT:
                array = makeObjectArray(length, JCSystem.CLEAR_ON_RESET);
                break;
            case JCSystem.ARRAY_TYPE_INT:
                SystemException.throwIt(SystemException.ILLEGAL_VALUE);
                break;
            default:
                SystemException.throwIt(SystemException.ILLEGAL_VALUE);
        }

        return array;
    }

    /**
     * @param theObj the object being queried
     * @return <code>NOT_A_TRANSIENT_OBJECT</code>, <code>CLEAR_ON_RESET</code>, or <code>CLEAR_ON_DESELECT</code>
     * @see javacard.framework.JCSystem#isTransient(Object)
     */
    public byte isTransient(Object theObj) {
        if (clearOnDeselect.contains(theObj)) {
            return JCSystem.CLEAR_ON_DESELECT;
        } else if (clearOnReset.contains(theObj)) {
            return JCSystem.CLEAR_ON_RESET;
        } else {
            return JCSystem.NOT_A_TRANSIENT_OBJECT;
        }
    }

    /**
     * Store <code>arrayRef</code> in memory depends by event type
     *
     * @param arrayRef array reference
     * @param event    event type
     */
    protected void storeArray(Object arrayRef, byte event) {
        add(arrayRef, event);
        switch (event) {
            case JCSystem.CLEAR_ON_DESELECT:
                clearOnDeselect.add(arrayRef);
                break;
            case JCSystem.CLEAR_ON_RESET:
                clearOnReset.add(arrayRef);
                break;
            default:
                SystemException.throwIt(SystemException.ILLEGAL_VALUE);
        }
    }

    /**
     * Zero <code>CLEAR_ON_DESELECT</code> buffers
     */
    protected void clearOnDeselect() {
        zero(clearOnDeselect);
    }

    /**
     * Zero <code>CLEAR_ON_RESET</code> and <code>CLEAR_ON_DESELECT</code>
     * buffers
     */
    protected void clearOnReset() {
        zero(clearOnDeselect);
        zero(clearOnReset);
    }

    /**
     * Perform <code>clearOnReset</code> and forget all buffers
     */
    protected void forgetBuffers() {
        clearOnReset();
        clearOnDeselect.clear();
        clearOnReset.clear();
        sumCOD = 0;
        sumCOR = 0;
    }

    /**
     * Zero all arrays in list
     *
     * @param list list of arrays
     */
    protected void zero(List<Object> list) {
        for (Object obj : list) {
            if (obj instanceof byte[]) {
                Arrays.fill((byte[]) obj, (byte) 0);
            } else if (obj instanceof short[]) {
                Arrays.fill((short[]) obj, (short) 0);
            } else if (obj instanceof Object[]) {
                Arrays.fill((Object[]) obj, null);
            } else if (obj instanceof boolean[]) {
                Arrays.fill((boolean[]) obj, false);
            } else {
                log.warn("Unsupported object: {}", obj.getClass());
            }
        }
    }

    public int getSumCOD() {
        return sumCOD;
    }

    public int getSumCOR() {
        return sumCOR;
    }


    /**
     * @return The current implementation always returns 32767
     * @see javacard.framework.JCSystem#getAvailableMemory(byte)
     */
    public short getAvailableTransientResetMemory() {
        // TODO: do some math
        return Short.MAX_VALUE;
    }

    /**
     * @return The current implementation always returns 32767
     * @see javacard.framework.JCSystem#getAvailableMemory(byte)
     */
    public short getAvailableTransientDeselectMemory() {
        // TODO: do some math
        return Short.MAX_VALUE;
    }

}
